"use client"

import { errorToStringFormatter } from "@/lib/formatters/error-formatters"
import { cn } from "@/lib/utils"
import { ComponentWithClassName, TODO } from "@/types/utils"
import { ReactNode } from "react"
import { SWRResponse } from "swr/_internal"
import LoadingDots from "./icons/loading.dots"

interface ClientDataWrapperProps<TData extends TODO> {
  swr: SWRResponse<TData>,
  content: ReactNode | ((data: NonNullable<TData>) => ReactNode),
  overrides?: {
    loading?: ReactNode,
    error?: ReactNode | ((error: TODO) => ReactNode),
    fallback?: ReactNode,
  }
}

/**
 * React component to wrap content based react components.
 * Requires swr to fetch content from a server or for long running tasks.
 * Allows for rendering data states on the client (e.g., loading, error, no data, etc.).
 * 
 * Overrides expose props to update state content. Set to `null` to render no content.
 */
export function ClientDataWrapper<TData extends TODO>({
  content: content,
  className,
  swr: swr,
  overrides,
}: ComponentWithClassName<ClientDataWrapperProps<TData>>) {
  const { isLoading, error, data } = swr
  const loading = overrides?.loading;
  const errorContent = overrides?.error;
  const fallback = overrides?.fallback;

  if (!isLoading && !error && !data) {
    return (
      typeof loading !== "undefined" ? (
        loading
      ) : (
        <div className={cn("p-4 grid place-content-center", className)}>
          <LoadingDots />
        </div>
      )
    )
  }

  if (!!error) {
    return (
      typeof errorContent === 'function'
        ? (
          errorContent(error)
        ) : typeof errorContent !== 'undefined'
          ? (
            errorContent
          ) : (
            <div className={cn("p-4 grid place-content-center", className)}>
              {errorToStringFormatter(error)}
            </div>
          )
    )
  }

  if (isLoading) {
    return (
      typeof loading !== "undefined" ? (
        loading
      ) : (
        <div className={cn("p-4 grid place-content-center", className)}>
          <LoadingDots />
        </div>
      )
    )
  }

  if (!data) {
    return (
      typeof fallback !== "undefined" ? (
        fallback
      ) : (
        <div className={cn("p-4 grid place-content-center", className)}>
          <LoadingDots />
        </div>
      )
    )
  }

  return (
    typeof content === "function"
      ? (
        content(data)
      ) : (
        content
      )
  )
}
